from flask import Flask, render_template, redirect, request, url_for, jsonify
from flask_socketio import SocketIO, emit
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
from werkzeug.security import generate_password_hash, check_password_hash
import os
import ssl
from datetime import datetime
import random

app = Flask(__name__)
app.config['SECRET_KEY'] = str(random.randint(1, 999999999))  # Change this to a secure random value
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///chat.db'
db = SQLAlchemy(app)
socketio = SocketIO(app)

login_manager = LoginManager(app)
login_manager.login_view = 'login'

class User(UserMixin, db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(200), nullable=False)
    color = db.Column(db.String(20), nullable=False)

class Message(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    user_id = db.Column(db.Integer, db.ForeignKey('user.id'), nullable=False)
    content = db.Column(db.Text, nullable=False)
    timestamp = db.Column(db.DateTime, nullable=False)

    user = db.relationship('User', backref=db.backref('messages', lazy=True))

with app.app_context():
    db.create_all()

def generate_color():
    r = random.randint(100, 200)
    g = random.randint(100, 200)
    b = random.randint(100, 200)
    return f"rgb({r},{g},{b})"

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

@app.route('/')
def index():
    if current_user.is_authenticated:
        return render_template('chat.html')
    return redirect(url_for('login'))

@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form.get('username')
        password = request.form.get('password')
        user = User.query.filter_by(username=username).first()
        if user and check_password_hash(user.password_hash, password):
            login_user(user)
            return redirect(url_for('index'))
        return 'Invalid username or password', 401
    return render_template('login.html')

@app.route('/logout')
@login_required
def logout():
    logout_user()
    return redirect(url_for('login'))

@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'POST':
        username = request.form.get('username')
        password = request.form.get('password')
        existing_user = User.query.filter_by(username=username).first()
        if existing_user:
            return 'Username already exists', 400
        new_user = User(username=username, password_hash=generate_password_hash(password), color=generate_color())
        db.session.add(new_user)
        db.session.commit()
        return redirect(url_for('login'))
    return render_template('register.html')

@socketio.on('connect')
@login_required
def handle_connect():
    emit('user_name', {'user_name': current_user.username, 'color': current_user.color})

@socketio.on('message')
@login_required
def handle_message(data):
    message = Message(user_id=current_user.id, content=data['msg'], timestamp=datetime.utcnow())
    db.session.add(message)
    db.session.commit()
    emit('message', {'user_name': current_user.username, 'msg': data['msg'], 'color': current_user.color}, broadcast=True)

@app.route('/get_messages')
@login_required
def get_messages():
    messages = Message.query.order_by(Message.timestamp).all()
    return jsonify([{
        'user_name': m.user.username,
        'msg': m.content,
        'color': m.user.color
    } for m in messages])

if __name__ == '__main__':
    context = ssl.SSLContext(ssl.PROTOCOL_TLS_SERVER)
    context.load_cert_chain('cert.pem', 'key.pem')  # Load SSL certificate and private key
    socketio.run(app, host='0.0.0.0', port=5000, ssl_context=context, debug=False, allow_unsafe_werkzeug=True)